# Bug Report: `Paragraph.set()` does not preserve original indentation

## Summary

When using `Paragraph.set()` to update a field value, the original indentation formatting is lost and replaced with a single-space indentation, despite deb822-lossless being designed to preserve formatting.

## Repository

`deb822-lossless` (https://github.com/jelmer/deb822-lossless)

## Problem Description

The `Entry::new()` method hardcodes a single space (`" "`) for the `INDENT` token on continuation lines, regardless of what the original indentation was in the parsed document. This causes formatting information to be lost when `Paragraph.set()` replaces an existing field.

## Affected Code

In `src/lossless.rs`, the `Entry::new()` method:

```rust
pub fn new(key: &str, value: &str) -> Entry {
    let mut builder = GreenNodeBuilder::new();

    builder.start_node(ENTRY.into());
    builder.token(KEY.into(), key);
    builder.token(COLON.into(), ":");
    builder.token(WHITESPACE.into(), " ");
    for (i, line) in value.split('\n').enumerate() {
        if i > 0 {
            builder.token(INDENT.into(), " ");  // <-- HARDCODED SINGLE SPACE
        }
        builder.token(VALUE.into(), line);
        builder.token(NEWLINE.into(), "\n");
    }
    builder.finish_node();
    Entry(SyntaxNode::new_root_mut(builder.finish()))
}
```

The `Paragraph.set()` method calls `Entry::new()` when replacing a field:

```rust
pub fn set(&mut self, key: &str, value: &str) {
    let new_entry = Entry::new(key, value);  // <-- Uses hardcoded indentation

    // Check if the field already exists and replace it
    for entry in self.entries() {
        if entry.key().as_deref() == Some(key) {
            self.0.splice_children(
                entry.0.index()..entry.0.index() + 1,
                vec![new_entry.0.into()],
            );
            return;
        }
    }
    // ...
}
```

## Steps to Reproduce

```rust
use deb822_lossless::Paragraph;

let original = r#"Source: example
Build-Depends: foo,
               bar,
               baz
"#;

let mut para: Paragraph = original.parse().unwrap();

// Modify the Build-Depends field
para.set("Build-Depends", "foo,\nbar,\nbaz");

println!("{}", para.to_string());
```

## Expected Behavior

The output should preserve the original 15-space indentation:

```
Source: example
Build-Depends: foo,
               bar,
               baz
```

## Actual Behavior

The output uses single-space indentation:

```
Source: example
Build-Depends: foo,
 bar,
 baz
```

## Impact

This breaks the "lossless" promise of the library. Tools that parse, modify, and write back deb822 files cannot preserve the original formatting, which can:

1. Create noisy diffs that obscure the actual changes
2. Violate Debian packaging conventions (which typically use 15-space indentation for continuation lines in control files)
3. Break automated tools that rely on formatting preservation

## Suggested Fix

`Entry::new()` should accept an optional indentation parameter, and `Paragraph.set()` should attempt to preserve the original indentation from the existing entry being replaced:

1. Extract the indentation from the existing entry (if one exists)
2. Pass it to a new constructor like `Entry::with_indentation(key, value, indent)`
3. Fall back to a reasonable default (like `" "` for single-line fields, or extract from neighboring fields for multi-line fields)

Alternatively, provide a separate method like `Entry::from_value_preserving_format(existing_entry, new_value)` that maintains the formatting structure of the original entry while only updating the value content.

## Workaround

Currently, users need to either:
1. Accept the formatting changes
2. Manually parse and reconstruct entries with correct indentation
3. Use `wrap_and_sort()` or similar methods to reformat consistently (but this may not match the original style)

## Version Information

Observed in deb822-lossless versions:
- 0.1.25
- 0.1.27
- 0.2.1
- 0.2.4
- 0.3.0
- 0.4.0
- 0.4.1

(All versions checked show the same hardcoded `" "` indentation)
