/* $Id: textcache.h $ */
/** @file
 * VBox Remote Desktop Protocol.
 */

/*
 * Copyright (C) 2006-2025 Oracle and/or its affiliates.
 *
 * This file is part of VirtualBox base platform packages, as
 * available from https://www.virtualbox.org.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation, in version 3 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <https://www.gnu.org/licenses>.
 *
 * SPDX-License-Identifier: GPL-3.0-only
 */
#ifndef VRDP_INCLUDED_SRC_textcache_h
#define VRDP_INCLUDED_SRC_textcache_h
#ifndef RT_WITHOUT_PRAGMA_ONCE
# pragma once
#endif

#include "vrdpdefs.h"
#include <VBox/RemoteDesktop/VRDEOrders.h>


/* textcache.cpp contains explanation about these values. */
#define VRDP_TC_NUM_FONTS  7
#define VRDP_TC_NUM_GLYPHS 0xfe

/* Text order flags. */
#define VRDP_TEXT2_FLAG_DEFAULT_PLACEMENT        0x00000001
#define VRDP_TEXT2_HORIZONTAL                    0x00000002
#define VRDP_TEXT2_VERTICAL                      0x00000004
#define VRDP_TEXT2_REVERSED                      0x00000008
#define VRDP_TEXT2_ZERO_BEARINGS                 0x00000010
#define VRDP_TEXT2_CHAR_INC_EQUAL_BM_BASE        0x00000020
#define VRDP_TEXT2_MAXEXT_EQUAL_BM_SIDE          0x00000040
#define VRDP_TEXT2_DO_NOT_SUBSTITUTE_DEVICE_FONT 0x00000080


/*
 * Per client text cache structures.
 *
 * Used for tracking, which glyphs have been already cached on the client.
 */

typedef struct _TCCLIENTGLYPH
{
    /* Whether the glyph was cached on the client. */
    uint32_t fCached : 1;
} TCCLIENTGLYPH;

typedef struct _TCCLIENTFONT
{
    /* Which cache generation is cached on the client. 0 = nothing cached. */
    uint32_t u32Uniq;

    /* Information about glyphs cached on the client. */
    TCCLIENTGLYPH aGlyphs[VRDP_TC_NUM_GLYPHS];
} TCCLIENTFONT;

typedef struct _TCCLIENTCACHE
{
    TCCLIENTFONT aFonts[VRDP_TC_NUM_FONTS];
} TCCLIENTCACHE;


/*
 * The server text cache.
 *
 * Assigns the glyphs to cache slots.
 *
 */

/* The forward declaration of the cache structure. */
struct _TEXTCACHE;
typedef struct _TEXTCACHE *PTEXTCACHE;

/* Description of a glyph in the cache. */
typedef struct _TCCACHEDGLYPH
{
    /* Glyphs TC_GF_* flags. */
    uint32_t u32Flags;

    /* The RDP glyph index in the font. */
    uint8_t u8Index;

    /* The unique glyph handle generated by the application (possible simple CRC64). */
    uint64_t u64Handle;

    /* The glyph bitmap dimensions. Note w == h == 0 for the space character. */
    uint16_t w;
    uint16_t h;

    /* The character origin in the bitmap. */
    int16_t  xOrigin;
    int16_t  yOrigin;

    /* 1BPP bitmap. Rows are byte aligned. Size is (((w + 7)/8) * h + 3) & ~3. */
    uint8_t *pu8Bitmap;

} TCCACHEDGLYPH;


/*
 * The text cache/draw order information.
 *
 * Contains backlinks to the server glyph cache.
 */

/* The glyph information. */
typedef struct _TCGLYPH
{
    /* The pointer to the glyph information in the main server glyph cache. */
    const TCCACHEDGLYPH *pCachedGlyph;

    /* Glyph position on the screen.
     * Only valid when (u8Flags & VRDP_TEXT2_CHAR_INC_EQUAL_BM_BASE) == 0
     */
    int16_t x;
    int16_t y;
} TCGLYPH;

/* FONTCACHE/TEXT2 sequence be sent to clients. */
typedef struct _TCFONTTEXT2
{
    /* The text cache that generated this sequence. */
    PTEXTCACHE ptc;

    /* The unique server font generation number for this structure.
     * Filled by the cache, client caches must verify their glyphs.
     */
    uint32_t u32Uniq;

    /* The server index for this font. */
    int index;

    /* The RDP font handle for those glyphs. */
    uint8_t u8RDPFontHandle;

    /* The VRDP_TEXT2_* flags. */
    uint8_t u8Flags;

    /* The RDP flags. */
    uint8_t u8CharInc;

    uint32_t rgbFG;
    uint32_t rgbBG;

    VRDEORDERRECT bkground;
    VRDEORDERRECT opaque;
    VRDEORDERPOINT origin;

    /* Glyphs information. */
    int cGlyphs;
    TCGLYPH aGlyphs[VRDP_TC_NUM_GLYPHS];
} TCFONTTEXT2;


PTEXTCACHE TCCreate (void);
void TCDelete (PTEXTCACHE ptc);

bool TCCacheGlyphs (PTEXTCACHE ptc, const VRDEORDERTEXT *pOrder, TCFONTTEXT2 **ppFontText2);
void TCFreeFontText2 (TCFONTTEXT2 *pFontText2);

#endif /* !VRDP_INCLUDED_SRC_textcache_h */
